#include "TrophiesSony.h"
#include "SignInSony.h"
#include "ErrorCodesSony.h"
#include "MessagePipe.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;


#define NP_SUPPORTS_TROPHY_GROUP_DETAILS !defined(SN_TARGET_PS3)

namespace UnityPlugin
{
	CachedTrophies gTrophies;

	DO_EXPORT( bool, PrxTrophyGetLastError ) (ResultCode* result)
	{
		return gTrophies.GetLastError(result);
	}

	DO_EXPORT( bool, PrxTrophyRegisterPackIsBusy ) ()
	{
		return gTrophies.RegisterTrophyPackIsBusy();
	}

	DO_EXPORT( bool, PrxTrophyRefreshGroupInfoIsBusy ) ()
	{
		return gTrophies.RefreshGroupInfoIsBusy();
	}

	DO_EXPORT( bool, PrxTrophyRefreshTrophyInfoIsBusy ) ()
	{
		return gTrophies.RefreshTrophyInfoIsBusy();
	}

	DO_EXPORT( bool, PrxTrophyRefreshProgressIsBusy ) ()
	{
		return gTrophies.RefreshProgressIsBusy();
	}

	DO_EXPORT( ErrorCode, PrxTrophyGetGameInfo ) (TrophyGameInfo* info)
	{
		return gTrophies.GetGameInfo(info);
	}

	DO_EXPORT( void, PrxTrophyLockList ) ()
	{
		gTrophies.Lock();
	}

	DO_EXPORT( void, PrxTrophyUnlockList ) ()
	{
		gTrophies.Unlock();
	}

	DO_EXPORT( ErrorCode, PrxTrophyRefreshGroupInfo ) ()
	{
		return gTrophies.RefreshGroupInfo();
	}

	DO_EXPORT( int, PrxTrophyGetGroupDetailsCount ) ()
	{
		return gTrophies.GetGroupDetailsCount();
	}

	DO_EXPORT( ErrorCode, PrxTrophyGetGroupDetails ) (int index, GroupDetails* details)
	{
		return gTrophies.GetGroupDetails(index, details);
	}

	DO_EXPORT( int, PrxTrophyGetGroupDataCount ) ()
	{
		return gTrophies.GetGroupDataCount();
	}

	DO_EXPORT( ErrorCode, PrxTrophyGetGroupData ) (int index, GroupData* data)
	{
		return gTrophies.GetGroupData(index, data);
	}

	DO_EXPORT( ErrorCode, PrxTrophyRefreshTrophyInfo ) ()
	{
		return gTrophies.RefreshTrophyInfo();
	}

	DO_EXPORT( int, PrxTrophyGetTrophyDetailsCount ) ()
	{
		return gTrophies.GetTrophyDetailsCount();
	}

	DO_EXPORT( ErrorCode, PrxTrophyGetTrophyDetails ) (int index, TrophyDetails* details)
	{
		return gTrophies.GetTrophyDetails(index, details);
	}

	DO_EXPORT( int, PrxTrophyGetTrophyDataCount ) ()
	{
		return gTrophies.GetTrophyDataCount();
	}

	DO_EXPORT( ErrorCode, PrxTrophyGetTrophyData ) (int index, TrophyData* data)
	{
		return gTrophies.GetTrophyData(index, data);
	}

	DO_EXPORT( ErrorCode, PrxTrophyRefreshProgress) ()
	{
		return gTrophies.RefreshProgress();
	}

	DO_EXPORT( ErrorCode, PrxTrophyGetProgress ) (TrophyProgress* info)
	{
		return gTrophies.GetProgress(info);
	}

	DO_EXPORT( ErrorCode, PrxTrophyAward ) (int index)
	{
		return gTrophies.AwardTrophy(index);
	}

	DO_EXPORT( ErrorCode, PrxRegisterTrophyPack ) ()
	{
		return gTrophies.RegisterTrophyPack(false);
	}


	struct PNG
	{
		char png[4];
		char crlfczlf[4];
	};
	struct IHDR
	{
		char ihdr[4];
		int ihdrlen;
		int width;
		int height;
		char bitDepth;
		char colorType;
		char compressionMethod;
		char filterMethod;
		char interlaceMethod;
	};

	void swapBytes(short* val)
	{
		char* bytes = (char*)val;
		char tmp = bytes[0]; bytes[0] = bytes[1]; bytes[1] = tmp;
	}

	void swapEndian(int* val)
	{
		short* words = (short*)val;
		short tmp = words[0]; words[0] = words[1]; words[1] = tmp;
		swapBytes(&words[0]);
		swapBytes(&words[1]);
	}

	CachedTrophies::CachedTrophies()
		: m_LastResult("Trophies")
		,m_RegisterTrophyPackBusy(false)
		,m_RefreshGroupInfoBusy(false)
		,m_RefreshTrophyInfoBusy(false)
		,m_RefreshProgressBusy(false)
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		,m_FutureGroupInfo(NULL)
#endif
		,m_GroupInfoGetCount(0)
		,m_FutureTrophyInfo(NULL)
		,m_FirstTimeTrophySetup(true)
	{
	}

	bool CachedTrophies::RegisterTrophyPackIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RegisterTrophyPackBusy;
	}

	bool CachedTrophies::RefreshGroupInfoIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RefreshGroupInfoBusy;
	}

	bool CachedTrophies::RefreshTrophyInfoIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RefreshTrophyInfoBusy;
	}

	bool CachedTrophies::RefreshProgressIsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_RefreshProgressBusy;
	}

	bool CachedTrophies::ProcessEvent(const Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
			case Event::trophyNotInit:				// An event generated when the trophy service is not initialized.
				handled = true;
				m_LastResult.SetResult(NP_ERR_FAILED, event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_TrophyError);
				break;

			case Event::trophyInvalidID:			// An event generated when an invalid trophy ID was passed to the trophy service.
				handled = true;
				m_LastResult.SetResult(NP_ERR_FAILED, event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_TrophyError);
				break;

			case Event::trophyAlreadyRegistered:	// An event generated when a trophy set is already registered.
				m_RegisterTrophyPackBusy = false;
				handled = true;
				Messages::LogWarning("trophyAlreadyRegistered\n");
				break;
			case Event::trophyNotEnoughSpace:		// An event generated when there is not enough space returned from trying to register trophy set.
			case Event::trophySetSetupFail:			// An event generated when a trophy set setup failed.
				m_RegisterTrophyPackBusy = false;
				handled = true;
				Messages::LogWarning("Trophy setup failed, error=%x", event.returnCode);
				m_LastResult.SetResult(NP_ERR_FAILED, event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_TrophySetSetupFail);
				break;

				// this is event is NEVER triggered for PS4
			case Event::trophyCacheReady:			// An event generated when the trophy cache is ready to use 
				{
					// 2nd step in the trophy init chain, next get the trophy game info.
					int ret = Trophy::Interface::trophyRetrieveGame(&m_FutureGameInfo, true);
					if (ret != SCE_TOOLKIT_NP_SUCCESS)
					{
						m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
					}
					handled = true;
				}
				break;

			case Event::trophyListAlreadyRetrieving:		// An event generated when the trophy service is already in the process of trying to retrieve another list.
				handled = true;
				Messages::LogWarning("trophyListAlreadyRetrieving");
				break;

			case Event::trophyAlreadyUnlocked:				// An event generated when a trophy is already unlocked.
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_TrophyUnlockedAlready);
				break;

			case Event::trophyContextFail:					// An event generated when there is a context error for the trophy service.
			case Event::trophyBusy:							// An event generated when the trophy service is busy and therefore cannot process a request.
			case Event::trophyTerminated:					// An event generated when the trophy service has been terminated.
				Messages::LogWarning("Trophy event not handled: event=%d\n", event.event);
				handled = true;
				break;

			case Event::trophySetSetupSuccess:
				// Do nothing, just the 1st step in the trophy init chain.
				handled = true;
				break;

			case Event::trophyGameInfoRetrievalSuccess:		// An event generated when trophy game information retrieval was successful.
				// 3rd and final step in the trophy init chain, cache the game info.
				if(m_FutureGameInfo.hasResult())
				{
					m_CachedGameInfo.numGroups = 0;
					m_CachedGameInfo.numTrophies = m_FutureGameInfo.get()->gameDetail.numTrophies;
					m_CachedGameInfo.numPlatinum = m_FutureGameInfo.get()->gameDetail.numPlatinum;
					m_CachedGameInfo.numGold = m_FutureGameInfo.get()->gameDetail.numGold;
					m_CachedGameInfo.numSilver = m_FutureGameInfo.get()->gameDetail.numSilver;
					m_CachedGameInfo.numBronze = m_FutureGameInfo.get()->gameDetail.numBronze;
					m_GameInfoTitle = (char*)&m_FutureGameInfo.get()->gameDetail.title[0];
					m_GameInfoDesc = (char*)&m_FutureGameInfo.get()->gameDetail.description[0];
					m_CachedGameInfo.title = m_GameInfoTitle.c_str();
					m_CachedGameInfo.description = m_GameInfoDesc.c_str();
				}

				m_RegisterTrophyPackBusy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_TrophySetSetupSuccess);
				Messages::AddMessage(Messages::kNPToolKit_TrophyGotGameInfo);
				Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_NPInitialized);
				break;

			case Event::trophyGameInfoRetrievalFail:		// An event generated when trophy game information retrieval failed.
				m_RegisterTrophyPackBusy = false;
				handled = true;
				m_LastResult.SetResultSCE(m_FutureGameInfo.getError(), true, __FUNCTION__, __LINE__);
				// Trophy setup must have succeeded so we still need to do its message.
				Messages::AddMessage(Messages::kNPToolKit_TrophySetSetupSuccess);
				break;

#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
			case Event::trophyGroupInfoRetrievalSuccess:	//< An event generated when group information retrieval was successful.
				{
					Assert(m_GroupInfoGetCount > 0);
					int i = m_CachedGameInfo.numGroups - m_GroupInfoGetCount;
					//int groupId = m_FutureGroupInfo[i].get()->trophyGroupDetails.groupId;
					Assert(m_FutureGroupInfo[i].get()->trophyGroupDetails.groupId == i + 1);
					sce::Toolkit::NP::TrophyGroupInfo* info = m_FutureGroupInfo[i].get();

					CachedGroupDetails* details = new CachedGroupDetails;
					details->groupId = info->trophyGroupDetails.groupId;
					details->numTrophies = info->trophyGroupDetails.numTrophies;
					details->numPlatinum = info->trophyGroupDetails.numPlatinum;
					details->numGold = info->trophyGroupDetails.numGold;
					details->numSilver = info->trophyGroupDetails.numSilver;
					details->numBronze = info->trophyGroupDetails.numBronze;
					details->title = (char*)&info->trophyGroupDetails.title[0];
					details->description = (char*)&info->trophyGroupDetails.description[0];
					details->iconData = (void *)info->iconData;
					if(details->iconData)
					{
						details->iconDataSize = info->size;
						PNG* png = (PNG*)details->iconData;
						IHDR* header = (IHDR*)(png+1);
						int width = header->width;
						int height = header->height;
						swapEndian(&width);
						swapEndian(&height);
						details->iconWidth = width;
						details->iconHeight = height;
					}
					else
					{
						details->iconDataSize = 0;
						details->iconWidth = 0;
						details->iconHeight = 0;
					}
					m_CachedGroupDetails.push_back(details);

					CachedGroupData* data = new CachedGroupData;
					data->groupId = info->trophyGroupData.groupId;
					data->unlockedTrophies = info->trophyGroupData.unlockedTrophies;
					data->unlockedPlatinum = info->trophyGroupData.unlockedPlatinum;
					data->unlockedGold = info->trophyGroupData.unlockedGold;
					data->unlockedSilver = info->trophyGroupData.unlockedSilver;
					data->unlockedBronze = info->trophyGroupData.unlockedBronze;
					data->progressPercentage = info->trophyGroupData.progressPercentage;
					m_CachedGroupData.push_back(data);

					m_GroupInfoGetCount--;
					if(m_GroupInfoGetCount == 0)
					{
						DestroyGroupInfo();
						m_RefreshGroupInfoBusy = false;
						Messages::AddMessage(Messages::kNPToolKit_TrophyGotGroupInfo);
					}
					handled = true;
				}
				break;

			case Event::trophyGroupInfoRetrievalFail:		// An event generated when group information retrieval failed.
				DestroyGroupInfo();

				m_RefreshGroupInfoBusy = false;
				handled = true;
				m_LastResult.SetResultSCE(m_FutureGroupInfo->getError(), true, __FUNCTION__, __LINE__);
				break;
#endif
			case Event::trophyListRetrievalSuccess:			// An event generated when a trophy list was retrieved successfully.
				ClearTrophyInfoList();

				for(int i=0; i<m_CachedGameInfo.numTrophies; i++)
				{
					sce::Toolkit::NP::TrophyInfo* info = m_FutureTrophyInfo[i].get();
					
					CachedTrophyDetails* details = new CachedTrophyDetails;
					details->trophyId = info->trophyDetails.trophyId;
					details->trophyGrade = info->trophyDetails.trophyGrade;
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
					details->groupId = info->trophyDetails.groupId;
#else
					details->groupId = -1;
#endif
					details->hidden = info->trophyDetails.hidden;
					details->name = (char*)&info->trophyDetails.name[0];
					details->description = (char*)&info->trophyDetails.description[0];
					m_CachedTrophyDetails.push_back(details);

					CachedTrophyData* data = new CachedTrophyData;
					data->trophyId = info->trophyData.trophyId;
					data->unlocked = info->trophyData.unlocked;
					data->timestamp = 0;
					data->iconData = (void *)info->iconData;
					if(data->iconData)
					{
						data->iconDataSize = info->size;
						PNG* png = (PNG*)data->iconData;
						IHDR* header = (IHDR*)(png+1);
						int width = header->width;
						int height = header->height;
						data->iconWidth = width;
						data->iconHeight = height;
					}
					else
					{
						data->iconDataSize = 0;
						data->iconWidth = 0;
						data->iconHeight = 0;
					}
					m_CachedTrophyData.push_back(data);
				}

				DestroyTrophyInfo();

				m_RefreshTrophyInfoBusy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_TrophyGotTrophyInfo);
				break;

			case Event::trophyListRetrievalFail:			// An event generated when a trophy list failed to be retrieved.
				m_RefreshTrophyInfoBusy = false;
				handled = true;
				m_LastResult.SetResultSCE(m_FutureTrophyInfo->getError(), true, __FUNCTION__, __LINE__);

				// Need to destroy trophy info after m_FutureTrophyInfo is used above
				DestroyTrophyInfo();

				break;


			case Event::trophyProgressSuccess:				// An event generated when the progress of the user has been retrieved.
				if(m_FutureProgress.hasResult())
				{
					m_CachedProgress.unlockedTrophies = m_FutureProgress.get()->unlockedTrophies;
					m_CachedProgress.unlockedPlatinum = m_FutureProgress.get()->unlockedPlatinum;
					m_CachedProgress.unlockedGold = m_FutureProgress.get()->unlockedGold;
					m_CachedProgress.unlockedSilver = m_FutureProgress.get()->unlockedSilver;
					m_CachedProgress.unlockedBronze = m_FutureProgress.get()->unlockedBronze;
					m_CachedProgress.progressPercentage = 0;
				}

				m_RefreshProgressBusy = false;
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_TrophyGotProgress);
				break;

			case Event::trophyProgressFail:					// An event generated when the progress of the user failed to be retrieved
				m_RefreshProgressBusy = false;
				handled = true;
				m_LastResult.SetResultSCE(m_FutureProgress.getError(), true, __FUNCTION__, __LINE__);
				break;

			case Event::trophyPlatinumUnlocked:				// An event generated when a platinum trophy is unlocked.
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_TrophyUnlockedPlatinum);
				break;

			case Event::trophyUnlockSuccess:				// An event generated when a trophy was unlocked successfully.
				handled = true;
				Messages::AddMessage(Messages::kNPToolKit_TrophyUnlocked);
				break;

			case Event::trophyUnlockFail:					// An event generated when a trophy could not be unlocked.
				// On PS3 we get the trophyUnlockFail event if already unlocked and not the expected trophyAlreadyUnlocked, so check the return code.
				if(event.returnCode == SCE_NP_TROPHY_ERROR_ALREADY_UNLOCKED)
				{
					Messages::AddMessage(Messages::kNPToolKit_TrophyUnlockedAlready);
				}
				else
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TrophyUnlockFailed);
				}
				handled = true;
				break;

			default:
				Messages::LogWarning("Unexpected event from trophy service: event=%d\n", event.event);
				handled = true;
				break;
		}

		return handled;
	}

	void CachedTrophies::Lock()
	{
		m_Lock.Lock();
	}

	void CachedTrophies::Unlock()
	{
		m_Lock.Unlock();
	}

	ErrorCode CachedTrophies::RegisterTrophyPack(bool cacheIcons)
	{
		if(RegisterTrophyPackIsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		bool DoCacheIcons = cacheIcons;
	/*	static bool DoCacheIcons = false;
		if (cacheIcons == true)
		{
			DoCacheIcons = true;
		}*/

		m_LastResult.Reset();

		int ret = Trophy::Interface::trophyRegisterSet(true,true,0,0);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		SimpleLock::AutoLock lock(m_Lock);
		m_RegisterTrophyPackBusy = true;
		return m_LastResult.GetResult();
	}



	ErrorCode CachedTrophies::GetGameInfo(TrophyGameInfo* info)
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_LastResult.Reset();
		*info = m_CachedGameInfo;
		return m_LastResult.GetResult();
	}

	void CachedTrophies::CreateGroupInfo(int size)
	{
		DestroyGroupInfo();
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		m_FutureGroupInfo = new Future<sce::Toolkit::NP::TrophyGroupInfo>[size];
#endif
	}

	void CachedTrophies::DestroyGroupInfo()
	{
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		delete[] m_FutureGroupInfo;
		m_FutureGroupInfo = NULL;
#endif
	}

	void CachedTrophies::ClearGroupInfoList()
	{
		for(size_t i=0; i<m_CachedGroupDetails.size(); i++)
		{
			delete m_CachedGroupDetails[i];
		}
		m_CachedGroupDetails.clear();

		for( size_t i=0; i<m_CachedGroupData.size(); i++)
		{
			delete m_CachedGroupData[i];
		}
		m_CachedGroupData.clear();
	}

	ErrorCode CachedTrophies::RefreshGroupInfo()
	{
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		SimpleLock::AutoLock lock(m_Lock);

		if(m_RefreshGroupInfoBusy)
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		m_LastResult.Reset();

		if(m_CachedGameInfo.numGroups > 0)
		{
			CreateGroupInfo(m_CachedGameInfo.numGroups);
			ClearGroupInfoList();

			Assert(m_GroupInfoGetCount == 0);
			for (int groupId=0; groupId<m_CachedGameInfo.numGroups; groupId++)
			{
				int ret = Trophy::Interface::trophyRetrieveGroups(&m_FutureGroupInfo[groupId], groupId, true);
				if (ret != SCE_TOOLKIT_NP_SUCCESS)
				{
					return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
				}
				m_GroupInfoGetCount ++;
			}

			m_RefreshGroupInfoBusy = true;
		}
		else
		{
			ClearGroupInfoList();
			Messages::AddMessage(Messages::kNPToolKit_TrophyGotGroupInfo);
		}
		return m_LastResult.GetResult();
#else
		return m_LastResult.SetResult(NP_ERR_NOT_SUPPORTED, true, __FUNCTION__, __LINE__);
#endif
	}

	int CachedTrophies::GetGroupDetailsCount() const
	{
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		return m_CachedGroupDetails.size();
#else
		printf("Trophy Group Functions not available on PS3\n");
		return -1;
#endif
	}

	int CachedTrophies::GetGroupDataCount() const
	{
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		return m_CachedGroupData.size();
#else
		printf("Trophy Group Functions not available on PS3\n");
		return -1;
#endif
	}

	ErrorCode CachedTrophies::GetGroupDetails(unsigned int index, GroupDetails* info)
	{
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		if(index >= m_CachedGroupDetails.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		const CachedGroupDetails* cached = m_CachedGroupDetails[index];

		info->groupId = cached->groupId;
		info->numTrophies = cached->numTrophies;
		info->numPlatinum = cached->numPlatinum;
		info->numGold = cached->numGold;
		info->numSilver = cached->numSilver;
		info->numBronze = cached->numBronze;
		info->iconDataSize = cached->iconDataSize;
		info->iconWidth = cached->iconWidth;
		info->iconHeight = cached->iconHeight;
		info->iconData = cached->iconData;
		info->title = cached->title.c_str();
		info->description = cached->description.c_str();

		return m_LastResult.GetResult();
#else
		return m_LastResult.SetResult(NP_ERR_NOT_SUPPORTED, true, __FUNCTION__, __LINE__);
#endif
	}

	ErrorCode CachedTrophies::GetGroupData(unsigned int index, GroupData* info)
	{
#if NP_SUPPORTS_TROPHY_GROUP_DETAILS
		if(index >= m_CachedGroupDetails.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		const CachedGroupData* cached = m_CachedGroupData[index];
		info->groupId = cached->groupId;
		info->unlockedTrophies = cached->unlockedTrophies;
		info->unlockedPlatinum = cached->unlockedPlatinum;
		info->unlockedGold = cached->unlockedGold;
		info->unlockedSilver = cached->unlockedSilver;
		info->unlockedBronze = cached->unlockedBronze;
		info->progressPercentage = cached->progressPercentage;

		return m_LastResult.GetResult();
#else
		return m_LastResult.SetResult(NP_ERR_NOT_SUPPORTED, true, __FUNCTION__, __LINE__);
#endif
	}

	void CachedTrophies::CreateTrophyInfo(int size)
	{
		Assert(size>0);
		DestroyTrophyInfo();
		m_FutureTrophyInfo = new Future<sce::Toolkit::NP::TrophyInfo>[size];
	}

	void CachedTrophies::DestroyTrophyInfo()
	{
		delete[] m_FutureTrophyInfo;
		m_FutureTrophyInfo = NULL;
	}

	void CachedTrophies::ClearTrophyInfoList()
	{
		for(size_t i=0; i<m_CachedTrophyDetails.size(); i++)
		{
			delete m_CachedTrophyDetails[i];
		}
		m_CachedTrophyDetails.clear();

		for(size_t i=0; i<m_CachedTrophyData.size(); i++)
		{
			delete m_CachedTrophyData[i];
		}
		m_CachedTrophyData.clear();
	}

	ErrorCode CachedTrophies::RefreshTrophyInfo()
	{
		SimpleLock::AutoLock lock(m_Lock);

		if(m_RefreshTrophyInfoBusy)
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		m_LastResult.Reset();
		
		CreateTrophyInfo(m_CachedGameInfo.numTrophies);
		int ret = sce::Toolkit::NP::Trophy::Interface::trophyRetrieveList(m_FutureTrophyInfo, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_RefreshTrophyInfoBusy = true;
		return m_LastResult.GetResult();
	}

	int CachedTrophies::GetTrophyDetailsCount() const
	{
		return m_CachedTrophyDetails.size();
	}

	int CachedTrophies::GetTrophyDataCount() const
	{
		return m_CachedTrophyData.size();
	}

	ErrorCode CachedTrophies::GetTrophyDetails(unsigned int index, TrophyDetails* info)
	{
		if(index >= m_CachedTrophyDetails.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		const CachedTrophyDetails* cached = m_CachedTrophyDetails[index];
		info->trophyId = cached->trophyId;
		info->trophyGrade = cached->trophyGrade;
		info->groupId = cached->groupId;
		info->hidden = cached->hidden;
		info->name = cached->name.c_str();
		info->description = cached->description.c_str();

		return m_LastResult.GetResult();
	}

	ErrorCode CachedTrophies::GetTrophyData(unsigned int index, TrophyData* info)
	{
		if(index >= m_CachedTrophyDetails.size())
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		const CachedTrophyData* cached = m_CachedTrophyData[index];
		info->trophyId = cached->trophyId;
		info->unlocked = cached->unlocked;
		info->timestamp = cached->timestamp;
		info->iconDataSize = cached->iconDataSize;
		info->iconWidth = cached->iconWidth;
		info->iconHeight = cached->iconHeight;
		info->iconData = cached->iconData;

		return m_LastResult.GetResult();
	}

	ErrorCode CachedTrophies::RefreshProgress()
	{
		SimpleLock::AutoLock lock(m_Lock);

		if(m_RefreshProgressBusy)
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		m_LastResult.Reset();

		int ret = sce::Toolkit::NP::Trophy::Interface::trophyRetrieveProgress(&m_FutureProgress, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_RefreshProgressBusy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedTrophies::GetProgress(TrophyProgress* info)
	{
		SimpleLock::AutoLock lock(m_Lock);
		m_LastResult.Reset();
		*info = m_CachedProgress;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedTrophies::AwardTrophy(int index)
	{
		SimpleLock::AutoLock lock(m_Lock);

		if(index < 0 || index > m_CachedGameInfo.numTrophies)
		{
			return m_LastResult.SetResult(NP_ERR_INDEX_OUT_OF_RANGE, true, __FUNCTION__, __LINE__);
		}

		m_LastResult.Reset();

		int ret = Trophy::Interface::trophyUnlock(index, true);
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		return m_LastResult.GetResult();
	}


} //namespace UnityPlugin
